/* assets/js/lang.js
   Live i18n switcher for EN/RU with localStorage persistence.
   Updates any element that has [data-i18n="dot.separated.key"].
*/

(function () {
  // =======================
  // 1) DICTIONARIES
  // =======================
  const DICTS = {
    en: {
      ui: { lang: { en: "EN", ru: "RU" } },
      page: { 
	            name: "/ StartUp Costs",
	            subtitle: "Minimum and maximum costs for startup registration" },
      filters: {
        all: "All",
        law: "Law",
        idea: "Idea",
        business: "Business",
        investment: "Investment"
      },
summary: {
  note:
    "Approximate cost if you want to obtain a startup idea, establish a legal entity with full one-year support, office, and registration in the official registry. Additionally, based on this, you will obtain a residence permit.",
  know:
    "These are indicative costs for full startup registration. Want to know the exact amount for your case? — Contact us."
},

      btn: { contact: '<i class="bi bi-whatsapp"></i> Contact us' },
      items: {
        srl:        { h3: "max € 11 375 / min € 6 225",  span: "Establishment of a Legal Entity (SRL)" },
        auth_cap:   { h3: "max € 10 000",                span: "Authorized capital" },
        invest:     { h3: "€ 50 000",                    span: "Guaranteed contribution" },
        idea:       { h3: "Startup idea (design, documentation, submission)", span: "max € 24 500 / min € 18 000" },
        support:    { h3: "max € 6 500 / min € 4 000",   span: "Support" },
        accounting: { h3: "max € 4 000 / min € 2 500",   span: "Accounting contract" },
        accommodation: { h3: "max € 1 500 / min € 700",  span: "Accommodation real estate agent, paperwork, and contracts" },
        monthly:    { h3: "€ 8 670",                     span: "Monthly expenses (residence, insurance, research)" }
      }
    },

    ru: {
      ui: { lang: { en: "EN", ru: "RU" } },
      page: { 
	            name: "/ Затраты на стартап",
	            subtitle: "Минимальные и максимальные затраты на регистрацию стартапа" },
      filters: {
        all: "Все",
        law: "Право",
        idea: "Идея",
        business: "Бизнес",
        investment: "Инвестиции"
      },
summary: {
  note:
    "Примерная стоимость, если вы хотите получить идею, открыть юридическое лицо с полным сопровождением на год, офисом и оформить регистрацию в официальном реестре. Дополнительно на этом основании вы получите ВНЖ.",
  know:
    "Это примерные затраты на регистрацию стартапа. Хотите узнать точную сумму для вашего случая? — Свяжитесь с нами."
},

      btn: { contact: '<i class="bi bi-whatsapp"></i> Связаться с нами' },
      items: {
        srl:        { h3: "макс € 11 375 / мин € 6 225",  span: "Учреждение юридического лица (SRL)" },
        auth_cap:   { h3: "макс € 10 000",                span: "Уставной капитал" },
        invest:     { h3: "€ 50 000",                     span: "Гарантированный взнос" },
        idea:       { h3: "Идея стартапа (дизайн, документация, подача)", span: "макс € 24 500 / мин € 18 000" },
        support:    { h3: "макс € 6 500 / мин € 4 000",   span: "Сопровождение" },
        accounting: { h3: "макс € 4 000 / мин € 2 500",   span: "Договор на бухгалтерское сопровождение" },
        accommodation: { h3: "макс € 1 500 / мин € 700",  span: "Агент по жилью, оформление и договоры" },
        monthly:    { h3: "€ 8 670",                      span: "Ежемесячные расходы (проживание, страховка, исследования)" }
      }
    }
  };

  // =======================
  // 2) HELPERS
  // =======================
  const getFromDict = (path, obj) => {
    if (!path) return null;
    return path.split(".").reduce((acc, k) => (acc && acc[k] != null ? acc[k] : null), obj);
  };

  const applyLang = (lang) => {
    const dict = DICTS[lang] || DICTS.en;

    // Set <html lang="">
    document.documentElement.setAttribute("lang", lang);

    // Replace innerHTML/text for all nodes with data-i18n
    document.querySelectorAll("[data-i18n]").forEach((node) => {
      const key = node.getAttribute("data-i18n");
      const val = getFromDict(key, dict);
      if (val == null) return;

      // Support input/textarea placeholders if needed
      const tag = node.tagName;
      if (tag === "INPUT" || tag === "TEXTAREA") {
        if (node.hasAttribute("placeholder")) node.setAttribute("placeholder", val);
        else node.value = val;
      } else {
        node.innerHTML = val;
      }
    });

    // Toggle active button
    document.querySelectorAll(".lang-btn").forEach((btn) => {
      btn.classList.toggle("active", btn.getAttribute("data-lang") === lang);
    });

    // Persist
    try { localStorage.setItem("lang", lang); } catch (e) {}
  };

  // =======================
  // 3) INIT
  // =======================
  const init = () => {
    let lang = "en";
    try {
      lang = localStorage.getItem("lang") || "en";
    } catch (e) {}

    // Bind clicks
    document.querySelectorAll(".lang-btn").forEach((btn) => {
      btn.addEventListener("click", (ev) => {
        ev.preventDefault();
        const l = btn.getAttribute("data-lang") || "en";
        applyLang(l);
      });
    });

    // Initial render
    applyLang(lang);
  };

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", init);
  } else {
    init();
  }
})();
