// =========================================================
// app.js — ZANZARA (Screen 0 + Screen 1 + Embedded Test + Prices)
// =========================================================

let mainVideoContainer;
window.__videoBound = false;
window.__pricesInited = false;

/* ================= Embedded Test bootstrapping ================= */
// We expose window.__embeddedTest immediately so clicks never "do nothing" if init fails or is slow.
let __embeddedTestApi = null;
let __embeddedTestInitPromise = null;
let __embeddedTestInitError = null;

function __showEmbeddedTestError(message) {
  const host = document.getElementById("embedded-test");
  const stage = document.getElementById("embedded-test-stage");
  const titleEl = document.getElementById("embedded-test-title");

  if (titleEl) titleEl.textContent = "Embedded test error";
  if (stage) {
    stage.innerHTML = `
      <div style="padding:16px;border:1px solid rgba(0,0,0,.15);border-radius:14px;background:#fff">
        <div style="font-weight:600;margin-bottom:8px">Cannot open the test</div>
        <div style="font-size:14px;line-height:1.35;opacity:.8">${message}</div>
      </div>
    `;
  }
  if (host) {
    host.hidden = false;
    host.setAttribute("aria-hidden", "false");
  }
}

window.__embeddedTest = {
  open: (slideId) => {
    // If already initialized — open immediately.
    if (__embeddedTestApi) {
      __embeddedTestApi.open(slideId);
      return;
    }

    // If previous init failed — show error UI.
    if (__embeddedTestInitError) {
      __showEmbeddedTestError(
        "test.html failed to load. Check browser console for details. " +
          "Most common causes: running via file:// (fetch blocked) or missing test.html on the server."
      );
      return;
    }

    // Lazy init (first click) OR wait for ongoing init.
    if (!__embeddedTestInitPromise) __embeddedTestInitPromise = initEmbeddedTest();

    __embeddedTestInitPromise
      .then(() => {
        if (__embeddedTestApi) __embeddedTestApi.open(slideId);
      })
      .catch((err) => {
        __embeddedTestInitError = err || new Error("Unknown init error");
        console.error("[embedded-test] init failed:", __embeddedTestInitError);
        __showEmbeddedTestError(
          "test.html failed to load. If you open index.html directly from disk, run a local server instead."
        );
      });
  },

  close: () => {
    if (__embeddedTestApi) __embeddedTestApi.close();
  },
};

/* ================= PRELOAD ================= */
document.addEventListener("DOMContentLoaded", async () => {
  try {
    mainVideoContainer = document.querySelector(".main-video-container");
    if (!mainVideoContainer) throw new Error("Main video container not found");
    await preloadVideos();
  } catch (error) {
    console.error("Video initialization error:", error);
  }
});

async function preloadVideos() {
  const videoSources = ["videos/001_V.mp4", "videos/PAUSE.mp4"];
  const loading = document.querySelector(".loader");

  try {
    await Promise.all(
      videoSources.map(
        (src) =>
          new Promise((resolve) => {
            const v = document.createElement("video");
            v.preload = "auto";
            v.onloadeddata = () => resolve();
            v.onerror = () => resolve(); // don't hard-fail
            v.src = src;
          })
      )
    );
  } catch (e) {
    console.error("Video preload failed:", e);
  } finally {
    document.body.classList.add("loaded");
    setTimeout(() => loading && loading.remove(), 500);

    initializeVideos();
    initializeStartupScreen();

    // load embedded test once (pre-warm; window.__embeddedTest also supports lazy init)
    __embeddedTestInitPromise = initEmbeddedTest();
    __embeddedTestInitPromise.catch(console.error);

    // guard play button
    startPlayToggleGuard();
  }
}

/* ================= MAIN VIDEO (Play / Pause) ================= */
function initializeVideos() {
  const pauseVideo = document.getElementById("video-pause");
  const mainVideo = document.getElementById("video-file");
  const playToggle = document.getElementById("play-toggle");

  if (!pauseVideo || !mainVideo || !playToggle) return;
  if (window.__videoBound) return;
  window.__videoBound = true;

  showPauseLayer();

  const setBtnState = (isPlaying) => {
    playToggle.setAttribute("aria-label", isPlaying ? "Pause" : "Play");
    playToggle.classList.toggle("is-playing", isPlaying);
  };

  playToggle.addEventListener("click", () => {
    if (document.body.classList.contains("is-screen1-open")) return;

    if (mainVideo.ended) mainVideo.currentTime = 0;

    if (mainVideo.paused) {
      showMainLayer();
      mainVideo.play().catch(() => {});
      setBtnState(true);
    } else {
      mainVideo.pause();
      showPauseLayer();
      setBtnState(false);
    }
  });

  mainVideo.addEventListener("play", () => {
    showMainLayer();
    setBtnState(true);
  });

  mainVideo.addEventListener("pause", () => {
    if (!mainVideo.ended) {
      showPauseLayer();
      setBtnState(false);
    }
  });

  mainVideo.addEventListener("ended", () => {
    showPauseLayer();
    setBtnState(false);
  });

  function showMainLayer() {
    pauseVideo.pause();
    pauseVideo.currentTime = 0;
    pauseVideo.style.opacity = "0";
    gsap.to(".video-main", {
      opacity: 1,
      duration: 0.25,
      ease: "power1.out",
      pointerEvents: "auto",
    });
  }

  function showPauseLayer() {
    gsap.to(".video-main", {
      opacity: 0,
      duration: 0.25,
      ease: "power1.out",
      pointerEvents: "none",
    });
    pauseVideo.style.opacity = "1";
    pauseVideo.play().catch(() => {});
  }
}

/* ================= Screen 1 (Startup) ================= */
function initializeStartupScreen() {
  const openLink = document.getElementById("startup-open");

  const screens = document.getElementById("screens");
  const screen = document.getElementById("screen-startup");

  const menu = document.getElementById("startup-menu");
  const content = document.getElementById("startup-content");

  const btnGot = document.getElementById("btn-got");
  const btnDamage = document.getElementById("btn-damage");
  const btnClose = document.getElementById("btn-close");

  const gridGot = document.getElementById("grid-got");
  const gridDamage = document.getElementById("grid-damage");

  const mainVideo = document.getElementById("video-file");

  if (
    !openLink ||
    !screens ||
    !screen ||
    !menu ||
    !content ||
    !btnGot ||
    !btnDamage ||
    !btnClose ||
    !gridGot ||
    !gridDamage
  ) {
    console.warn("[startup] Required elements missing");
    return;
  }

  let isScreenOpen = false;
  let isDocked = false;
  let activeTab = null;

  // scroll block (prevents background scroll)
  const _blockScroll = (e) => e.preventDefault();

  function enableScrollBlock() {
    window.addEventListener("wheel", _blockScroll, { passive: false });
    window.addEventListener("touchmove", _blockScroll, { passive: false });
  }

  function disableScrollBlock() {
    window.removeEventListener("wheel", _blockScroll);
    window.removeEventListener("touchmove", _blockScroll);
  }

  // initial state
  screens.classList.remove("is-open");
  screens.setAttribute("aria-hidden", "true");

  gsap.set(screen, { xPercent: 110 });
  gsap.set(menu, { left: "50%", top: "50%", xPercent: -50, yPercent: -50 });

  // IMPORTANT: content starts disabled
  gsap.set(content, { autoAlpha: 0, pointerEvents: "none" });
  content.setAttribute("aria-hidden", "true");

  gsap.set(gridGot, { autoAlpha: 0, display: "none" });
  gsap.set(gridDamage, { autoAlpha: 0, display: "none" });

  menu.classList.remove("is-got", "is-damage");

  function openScreen(e) {
    e && e.preventDefault();
    if (isScreenOpen) return;

    try {
      mainVideo && mainVideo.pause();
    } catch {}

    isScreenOpen = true;
    document.body.classList.add("is-screen1-open");

    screens.classList.add("is-open");
    screens.setAttribute("aria-hidden", "false");
    enableScrollBlock();

    // ✅ CRITICAL: make content clickable (some CSS may block it)
    content.style.pointerEvents = "auto";
    content.setAttribute("aria-hidden", "false");

    // ✅ open with dock + GOT so the grid exists and is clickable
    dockMenu(true);
    setActiveTab("got", true);

    window.__embeddedTest?.close?.();

    gsap.killTweensOf(screen);
    gsap.set(screen, { xPercent: 110 });
    gsap.to(screen, { xPercent: 0, duration: 0.48, ease: "power2.out" });
  }

  function closeScreen() {
    if (!isScreenOpen) return;
    isScreenOpen = false;

    gsap.killTweensOf(screen);
    gsap.to(screen, {
      xPercent: 110,
      duration: 0.42,
      ease: "power2.in",
      onComplete: () => {
        screens.classList.remove("is-open");
        screens.setAttribute("aria-hidden", "true");

        disableScrollBlock();
        document.body.classList.remove("is-screen1-open");

        window.__embeddedTest?.close?.();

        // back to "closed" state
        content.setAttribute("aria-hidden", "true");
        content.style.pointerEvents = "none";
      },
    });
  }

  function dockMenu(immediate) {
    if (isDocked) return;
    isDocked = true;

    gsap.killTweensOf(menu);
    gsap.to(menu, {
      top: 38,
      yPercent: 0,
      duration: immediate ? 0 : 0.42,
      ease: "power2.out",
    });

    content.setAttribute("aria-hidden", "false");

    gsap.killTweensOf(content);
    gsap.to(content, {
      autoAlpha: 1,
      duration: immediate ? 0 : 0.18,
      ease: "power1.out",
      onStart: () => gsap.set(content, { pointerEvents: "auto" }),
    });
  }

  function undock(immediate) {
    isDocked = false;

    menu.classList.remove("is-got", "is-damage");

    content.setAttribute("aria-hidden", "true");
    gsap.set(content, { pointerEvents: "none" });
    gsap.killTweensOf(content);
    gsap.to(content, {
      autoAlpha: 0,
      duration: immediate ? 0 : 0.14,
      ease: "power1.out",
    });

    gsap.killTweensOf(menu);
    gsap.to(menu, {
      top: "50%",
      yPercent: -50,
      duration: immediate ? 0 : 0.38,
      ease: "power2.inOut",
    });

    setActiveTab(null, true);
    window.__embeddedTest?.close?.();
  }

  function setActiveTab(tab, immediate) {
    if (tab && activeTab === tab) return;
    activeTab = tab;

    menu.classList.remove("is-got", "is-damage");
    if (tab === "got") menu.classList.add("is-got");
    if (tab === "damage") menu.classList.add("is-damage");

    gsap.killTweensOf([gridGot, gridDamage]);

    const hideGrid = (el) => {
      if (immediate) {
        el.style.display = "none";
        el.setAttribute("aria-hidden", "true");
        gsap.set(el, { autoAlpha: 0 });
        return;
      }
      gsap.to(el, {
        autoAlpha: 0,
        duration: 0.15,
        ease: "power1.out",
        onComplete: () => {
          el.style.display = "none";
          el.setAttribute("aria-hidden", "true");
        },
      });
    };

    const showGrid = (el) => {
      el.style.display = "grid";
      el.setAttribute("aria-hidden", "false");
      if (immediate) {
        gsap.set(el, { autoAlpha: 1 });
        return;
      }
      gsap.fromTo(
        el,
        { autoAlpha: 0 },
        { autoAlpha: 1, duration: 0.2, ease: "power1.out" }
      );
    };

    window.__embeddedTest?.close?.();

    if (!tab) {
      hideGrid(gridGot);
      hideGrid(gridDamage);
      return;
    }

    if (tab === "got") {
      hideGrid(gridDamage);
      showGrid(gridGot);
    } else if (tab === "damage") {
      hideGrid(gridGot);
      showGrid(gridDamage);
    }
  }

  // wire buttons
  openLink.addEventListener("click", openScreen);
  btnClose.addEventListener("click", closeScreen);

  btnGot.addEventListener("click", () => {
    dockMenu(false);
    setActiveTab("got", false);
  });

  btnDamage.addEventListener("click", () => {
    dockMenu(false);
    setActiveTab("damage", false);
  });

  document.addEventListener("keydown", (e) => {
    if (e.key !== "Escape") return;
    if (!isScreenOpen) return;

    if (isDocked) undock(false);
    else closeScreen();
  });

  // Clicking any card in GOT grid opens embedded test
  const gotMap = [
    "slide1d1",
    "slide2d1",
    "slide3d1",
    "slide4d1",
    "slide5d1",
    "slide6d1",
  ];

  gridGot.addEventListener("click", (e) => {
    const card = e.target.closest(".startup-card");
    if (!card) return;

    e.preventDefault();

    const cards = Array.from(gridGot.querySelectorAll(".startup-card"));
    const idx = cards.indexOf(card);
    const slideId = gotMap[idx] || "slide1d1";

    window.__embeddedTest?.open(slideId);
  });
}

/* ================= Embedded Test (load from test.html) ================= */
async function initEmbeddedTest() {
  const host = document.getElementById("embedded-test");
  const stage = document.getElementById("embedded-test-stage");
  const backBtn = document.getElementById("embedded-test-back");
  const titleEl = document.getElementById("embedded-test-title");

  if (!host || !stage || !backBtn) {
    console.warn("[embedded-test] Required elements missing");
    return;
  }

  // reset init error flag on successful entry
  __embeddedTestInitError = null;

  // fetch test.html
  let res;
  try {
    res = await fetch("test.html", { cache: "no-cache" });
  } catch (e) {
    throw new Error('Fetch failed for "test.html" (network or file:// restriction).');
  }
  if (!res.ok) {
    throw new Error(`Failed to load test.html (HTTP ${res.status})`);
  }
  const html = await res.text();
  const doc = new DOMParser().parseFromString(html, "text/html");

  // take only slides
  const slides = Array.from(doc.querySelectorAll('.slide[id^="slide"]'));
  if (!slides.length) {
    console.warn("[embedded-test] No slides found in test.html");
    return;
  }

  // inject
  stage.innerHTML = "";
  slides.forEach((s) => {
    const clone = s.cloneNode(true);
    // neutralize index links if any
    clone
      .querySelectorAll('a[href^="index.html"]')
      .forEach((a) => a.setAttribute("href", "#"));
    stage.appendChild(clone);
  });

  // init prices ONCE (after slides exist)
  if (!window.__pricesInited && typeof window.initPrices === "function") {
    window.__pricesInited = true;
    window.initPrices("#embedded-test");
  }

  function showSlide(id) {
    stage.querySelectorAll(".slide").forEach((x) => x.classList.remove("is-active"));
    const target = stage.querySelector(`#${CSS.escape(id)}`);
    if (target) {
      target.classList.add("is-active");
      stage.scrollTop = 0;
      if (titleEl) titleEl.textContent = id;
    }
  }

  // delegated navigation via class "slide-1d2-1d3"
  stage.addEventListener("click", (e) => {
    const btn = e.target.closest("button, a");
    if (!btn) return;

    if (btn.tagName === "A") e.preventDefault();

    const cls = Array.from(btn.classList).find((c) => /^slide-\d+d\d+-\d+d\d+$/.test(c));
    if (!cls) return;

    const to = cls.split("-")[2];
    showSlide(`slide${to}`);
  });

  backBtn.addEventListener("click", () => {
    host.hidden = true;
    host.setAttribute("aria-hidden", "true");

    // return to GOT grid (if visible in Screen 1)
    const grid = document.getElementById("grid-got");
    if (grid) {
      grid.style.display = "grid";
      grid.setAttribute("aria-hidden", "false");
      gsap.set(grid, { autoAlpha: 1 });
    }
  });

  // public API
  __embeddedTestApi = {
    open: (slideId) => {
      const gridGot = document.getElementById("grid-got");
      const gridDamage = document.getElementById("grid-damage");

      if (gridGot) {
        gridGot.style.display = "none";
        gridGot.setAttribute("aria-hidden", "true");
        gsap.set(gridGot, { autoAlpha: 0 });
      }
      if (gridDamage) {
        gridDamage.setAttribute("aria-hidden", "true");
      }

      host.hidden = false;
      host.setAttribute("aria-hidden", "false");

      showSlide(slideId || slides[0].id);
    },

    close: () => {
      host.hidden = true;
      host.setAttribute("aria-hidden", "true");
    },
  };
}

/* ================= Play button visibility guard ================= */
function startPlayToggleGuard() {
  const btn = document.getElementById("play-toggle");
  const circle = document.querySelector(".main-video-container");
  if (!btn || !circle) return;

  const tick = () => {
    // if Screen 1 open — hide
    if (document.body.classList.contains("is-screen1-open")) {
      btn.style.opacity = 0;
      btn.style.pointerEvents = "none";
      requestAnimationFrame(tick);
      return;
    }

    const r = circle.getBoundingClientRect();
    const isSmall = r.width < 260 || r.height < 260;
    const isOff = r.right < 0 || r.left > window.innerWidth || r.bottom < 0 || r.top > window.innerHeight;

    if (isSmall || isOff) {
      btn.style.opacity = 0;
      btn.style.pointerEvents = "none";
    } else {
      btn.style.opacity = 1;
      btn.style.pointerEvents = "auto";
    }

    requestAnimationFrame(tick);
  };

  requestAnimationFrame(tick);
}
